/*
 * Decompiled with CFR 0.152.
 */
package picard.fingerprint;

import htsjdk.samtools.SamReader;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.file.Path;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.argparser.Hidden;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;
import picard.fingerprint.CappedHaplotypeProbabilities;
import picard.fingerprint.CrosscheckMetric;
import picard.fingerprint.Fingerprint;
import picard.fingerprint.FingerprintChecker;
import picard.fingerprint.FingerprintIdDetails;
import picard.fingerprint.HaplotypeMap;
import picard.fingerprint.HaplotypeProbabilities;
import picard.fingerprint.MatchResults;
import picard.util.TabbedInputParser;

@CommandLineProgramProperties(summary="Checks that all data in the set of input files appear to come from the same individual. Can be used to cross-check readgroups, libraries, samples, or files. Operates on bams/sams/crams and vcfs (including gvcfs). \n<h3>Summary</h3>\nChecks if all the genetic data within a set of files appear to come from the same individual. It quickly determines whether a group's genotype matches that of an input SAM/BAM/CRAM/VCF by selective sampling, and has been designed to work well for low-depth SAM/BAMs (as well as high depth ones and VCFs.) The tool collects fingerprints (essentially, genotype information from different parts of the genome) at the finest level available in the data (readgroup for SAM files and sample for VCF files) and then optionally aggregates it by library, sample or file, to increase power and provide results at the desired resolution. Output is in a \"Moltenized\" format, one row per comparison. The results are emitted into a CrosscheckMetric metric file. In this format the output will include the LOD score and also tumor-aware LOD score which can help assess identity even in the presence of a severe loss of heterozygosity with high purity (which could cause it to otherwise fail to notice that samples are from the same individual.) A matrix output is also available to facilitate visual inspection of crosscheck results.\n \nSince there can be many rows of output in the metric file, we recommend the use of ClusterCrosscheckMetrics as a follow-up step to running CrosscheckFingerprints.\n \nThere are cases where one would like to identify a few groups out of a collection of many possible groups (say to link a bam to it's correct sample in a multi-sample vcf. In this case one would not case for the cross-checking of the various samples in the VCF against each other, but only in checking the identity of the bam against the various samples in the vcf. The SECOND_INPUT is provided for this use-case. With SECOND_INPUT provided, CrosscheckFingerprints does the following:\n - aggregation of data happens independently for the input files in INPUT and SECOND_INPUT. \n - aggregation of data happens at the SAMPLE level \n - each samples from INPUT will only be compared to that same sample in SECOND_INPUT. \n - MATRIX_OUTPUT is disabled. \nIn some cases, the groups collected may not have any observations (calls for a vcf, reads for a bam) at fingerprinting sites, or a sample in INPUT may be missing from the SECOND_INPUT. These cases are handled as follows:  If running in CHECK_SAME_SAMPLES mode with INPUT and SECOND_INPUT, and either INPUT or SECOND_INPUT includes a sample not found in the other, or contains a sample with no observations at any fingerprinting sites, an error will be logged and the tool will return EXIT_CODE_WHEN_MISMATCH. In all other running modes, when any group which is being crosschecked does not have any observations at fingerprinting sites, a warning is logged.  As long as there is at least one comparison where both sides have observations at fingerprinting sites, the tool will return zero.  However, if all comparisons have at least one side with no observations at fingerprinting sites, an error will be logged and the tool will return EXIT_CODE_WHEN_NO_VALID_CHECKS.\n<hr/><h3>Examples</h3><h4>Check that all the readgroups from a sample match each other:</h4><pre>    java -jar picard.jar CrosscheckFingerprints \\\n          INPUT=sample.with.many.readgroups.bam \\\n          HAPLOTYPE_MAP=fingerprinting_haplotype_database.txt \\\n          LOD_THRESHOLD=-5 \\\n          OUTPUT=sample.crosscheck_metrics </pre>\n <h4>Check that all the readgroups match as expected when providing reads from two samples from the same individual:</h4> <pre>     java -jar picard.jar CrosscheckFingerprints \\\n           INPUT=sample.one.with.many.readgroups.bam \\\n           INPUT=sample.two.with.many.readgroups.bam \\\n           HAPLOTYPE_MAP=fingerprinting_haplotype_database.txt \\\n           LOD_THRESHOLD=-5 \\\n           EXPECT_ALL_GROUPS_TO_MATCH=true \\\n           OUTPUT=sample.crosscheck_metrics </pre>\n\n<h4>Detailed Explanation</h4>\nThis tool calculates the LOD score for identity check between \"groups\" of data in the INPUT files as defined by the CROSSCHECK_BY argument. A positive value indicates that the data seems to have come from the same individual or, in other words the identity checks out. The scale is logarithmic (base 10), so a LOD of 6 indicates that it is 1,000,000 more likely that the data matches the genotypes than not. A negative value indicates that the data do not match. A score that is near zero is inconclusive and can result from low coverage or non-informative genotypes. Each group is assigned a sample identifier (for SAM this is taken from the SM tag in the appropriate readgroup header line, for VCF this is taken from the column label in the file-header. After combining all the data from the same group together, an all-against-all comparison is performed. Results are categorized as one of EXPECTED_MATCH, EXPECTED_MISMATCH, UNEXPECTED_MATCH, UNEXPECTED_MISMATCH, or AMBIGUOUS depending on the LOD score and on whether the sample identifiers of the groups agree: LOD scores that are less than LOD_THRESHOLD are considered mismatches, and those greater than -LOD_THRESHOLD are matches (between is ambiguous). If the sample identifiers are equal, the groups are expected to match. They are expected to mismatch otherwise. \n\nThe identity check makes use of haplotype blocks defined in the HAPLOTYPE_MAP file to enable it to have higher statistical power for detecting identity or swap by aggregating data from several SNPs in the haplotype block. This enables an identity check of samples with very low coverage (e.g. ~1x mean coverage).\n \nWhen provided a VCF, the identity check looks at the PL, GL and GT fields (in that order) and uses the first one that it finds. ", oneLineSummary="Checks that all data in the input files appear to have come from the same individual", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class CrosscheckFingerprints
extends CommandLineProgram {
    @Argument(shortName="I", doc="One or more input files (or lists of files) with which to compare fingerprints.", minElements=1)
    public List<String> INPUT;
    @Argument(doc="A tsv with two columns representing the sample as it appears in the INPUT data (in column 1) and the sample as it should be used for comparisons to SECOND_INPUT (in the second column). Need only include the samples that change. Values in column 1 should be unique. Values in column 2 should be unique even in union with the remaining unmapped samples. Should only be used with SECOND_INPUT. ", optional=true, mutex={"INPUT_SAMPLE_FILE_MAP"})
    public File INPUT_SAMPLE_MAP;
    @Argument(doc="A tsv with two columns representing the sample as it should be used for comparisons to SECOND_INPUT (in the first column) and  the source file (in INPUT) for the fingerprint (in the second column). Need only to include the samples that change. Values in column 1 should be unique even in union with the remaining unmapped samples. Values in column 2 should be unique in the file. Will error if more than one sample is found in a file (multi-sample vcf) pointed to in column 2. Should only be used in the presence of SECOND_INPUT. ", optional=true, mutex={"INPUT_SAMPLE_MAP"})
    public File INPUT_SAMPLE_FILE_MAP;
    @Argument(shortName="SI", optional=true, mutex={"MATRIX_OUTPUT"}, doc="A second set of input files (or lists of files) with which to compare fingerprints. If this option is provided the tool compares each sample in INPUT with the sample from SECOND_INPUT that has the same sample ID. In addition, data will be grouped by SAMPLE regardless of the value of CROSSCHECK_BY. When operating in this mode, each sample in INPUT must also have a corresponding sample in SECOND_INPUT. If this is violated, the tool will proceed to check the matching samples, but report the missing samples and return a non-zero error-code.")
    public List<String> SECOND_INPUT;
    @Argument(doc="A tsv with two columns representing the sample as it appears in the SECOND_INPUT data (in column 1) and the sample as it should be used for comparisons to INPUT (in the second column). Note that in case of unrolling files (file-of-filenames) one would need to reference the final file, i.e. the file that contains the genomic data. Need only include the samples that change. Values in column 1 should be unique. Values in column 2 should be unique even in union with the remaining unmapped samples. Should only be used with SECOND_INPUT. ", optional=true)
    public File SECOND_INPUT_SAMPLE_MAP;
    @Argument(doc="An argument that controls how crosschecking with both INPUT and SECOND_INPUT should occur. ")
    public Fingerprint.CrosscheckMode CROSSCHECK_MODE = Fingerprint.CrosscheckMode.CHECK_SAME_SAMPLE;
    @Argument(shortName="O", optional=true, doc="Optional output file to write metrics to. Default is to write to stdout.")
    public File OUTPUT = null;
    @Argument(shortName="MO", optional=true, doc="Optional output file to write matrix of LOD scores to. This is less informative than the metrics output and only contains Normal-Normal LOD score (i.e. doesn't account for Loss of Heterozygosity). It is however sometimes easier to use visually.", mutex={"SECOND_INPUT"})
    public File MATRIX_OUTPUT = null;
    @Argument(shortName="H", doc="The file lists a set of SNPs, optionally arranged in high-LD blocks, to be used for fingerprinting. See https://software.broadinstitute.org/gatk/documentation/article?id=9526 for details.")
    public File HAPLOTYPE_MAP;
    @Argument(shortName="LOD", doc="If any two groups (with the same sample name) match with a LOD score lower than the threshold the tool will exit with a non-zero code to indicate error. Program will also exit with an error if it finds two groups with different sample name that match with a LOD score greater than -LOD_THRESHOLD.\n\nLOD score 0 means equal likelihood that the groups match vs. come from different individuals, negative LOD score -N, mean 10^N time more likely that the groups are from different individuals, and +N means 10^N times more likely that the groups are from the same individual. ")
    public double LOD_THRESHOLD = 0.0;
    @Argument(doc="Specificies which data-type should be used as the basic comparison unit. Fingerprints from readgroups can be \"rolled-up\" to the LIBRARY, SAMPLE, or FILE level before being compared. Fingerprints from VCF can be be compared by SAMPLE or FILE.")
    public CrosscheckMetric.DataType CROSSCHECK_BY = CrosscheckMetric.DataType.READGROUP;
    @Argument(doc="The number of threads to use to process files and generate fingerprints.")
    public int NUM_THREADS = 1;
    @Argument(doc="specifies whether the Tumor-aware result should be calculated. These are time consuming and can roughly double the runtime of the tool. When crosschecking many groups not calculating the tumor-aware  results can result in a significant speedup.")
    public boolean CALCULATE_TUMOR_AWARE_RESULTS = true;
    @Argument(doc="Allow the use of duplicate reads in performing the comparison. Can be useful when duplicate marking has been overly aggressive and coverage is low.")
    public boolean ALLOW_DUPLICATE_READS = false;
    @Argument(doc="Assumed genotyping error rate that provides a floor on the probability that a genotype comes from the expected sample. Must be greater than zero. ")
    public double GENOTYPING_ERROR_RATE = 0.01;
    @Argument(doc="If true then only groups that do not relate to each other as expected will have their LODs reported.")
    public boolean OUTPUT_ERRORS_ONLY = false;
    @Argument(doc="The rate at which a heterozygous genotype in a normal sample turns into a homozygous (via loss of heterozygosity) in the tumor (model assumes independent events, so this needs to be larger than reality).", optional=true)
    public double LOSS_OF_HET_RATE = 0.5;
    @Argument(doc="Expect all groups' fingerprints to match, irrespective of their sample names.  By default (with this value set to false), groups (readgroups, libraries, files, or samples) with different sample names are expected to mismatch, and those with the same sample name are expected to match. ")
    public boolean EXPECT_ALL_GROUPS_TO_MATCH = false;
    @Argument(doc="When one or more mismatches between groups is detected, exit with this value instead of 0.")
    public int EXIT_CODE_WHEN_MISMATCH = 1;
    @Argument(doc="When all LOD score are zero, exit with this value.")
    public int EXIT_CODE_WHEN_NO_VALID_CHECKS = 1;
    @Argument(doc="Maximal effect of any single haplotype block on outcome (-log10 of maximal likelihood difference between the different values for the three possible genotypes).", minValue=0.0)
    public double MAX_EFFECT_OF_EACH_HAPLOTYPE_BLOCK = 3.0;
    @Hidden
    @Argument(doc="When true code will check for readability on input files (this can be slow on cloud access)")
    public boolean TEST_INPUT_READABILITY = true;
    private final Log log = Log.getInstance(CrosscheckFingerprints.class);
    private double[][] crosscheckMatrix = null;
    private final List<String> lhsMatrixKeys = new ArrayList<String>();
    private final List<String> rhsMatrixKeys = new ArrayList<String>();

    @Override
    protected String[] customCommandLineValidation() {
        if (this.GENOTYPING_ERROR_RATE <= 0.0) {
            return new String[]{"GENOTYPING_ERROR_RATE must be greater than zero. Found " + this.GENOTYPING_ERROR_RATE};
        }
        if (this.GENOTYPING_ERROR_RATE >= 1.0) {
            return new String[]{"GENOTYPING_ERROR_RATE must be strictly less than 1, found " + this.GENOTYPING_ERROR_RATE};
        }
        if (this.SECOND_INPUT == null && this.INPUT_SAMPLE_MAP != null) {
            return new String[]{"INPUT_SAMPLE_MAP can only be used when also using SECOND_INPUT"};
        }
        if (this.SECOND_INPUT == null && this.SECOND_INPUT_SAMPLE_MAP != null) {
            return new String[]{"SECOND_INPUT_SAMPLE_MAP can only be used when also using SECOND_INPUT"};
        }
        if (this.REFERENCE_SEQUENCE == null) {
            ArrayList<String> allInputs = new ArrayList<String>(this.INPUT);
            allInputs.addAll(this.SECOND_INPUT);
            for (String input : allInputs) {
                if (!input.endsWith(SamReader.Type.CRAM_TYPE.fileExtension())) continue;
                return new String[]{"REFERENCE must be provided when using CRAM as input."};
            }
        }
        return super.customCommandLineValidation();
    }

    @Override
    protected int doWork() {
        int numUnexpected;
        IOUtil.assertFileIsReadable(this.HAPLOTYPE_MAP);
        if (this.OUTPUT != null) {
            IOUtil.assertFileIsWritable(this.OUTPUT);
        }
        if (!this.SECOND_INPUT.isEmpty() && this.CROSSCHECK_MODE == Fingerprint.CrosscheckMode.CHECK_SAME_SAMPLE) {
            this.log.info("SECOND_INPUT is not empty, and CROSSCHECK_MODE==CHECK_SAME_SAMPLE. NOT doing cross-check. Will only compare each SAMPLE in INPUT against that sample in SECOND_INPUT.");
            if (this.CROSSCHECK_BY != CrosscheckMetric.DataType.SAMPLE) {
                this.log.warn("CROSSCHECK_BY is not SAMPLE, This doesn't make sense in non-crosscheck mode. Setting CROSSCHECK_BY to SAMPLE.");
                this.CROSSCHECK_BY = CrosscheckMetric.DataType.SAMPLE;
            }
        }
        if (!this.SECOND_INPUT.isEmpty() && this.CROSSCHECK_MODE == Fingerprint.CrosscheckMode.CHECK_ALL_OTHERS) {
            this.log.info("SECOND_INPUT is not empty, and CROSSCHECK_MODE==CHECK_ALL_OTHERS. Will compare fingerprints from INPUT against all the fingerprints in SECOND_INPUT.");
        }
        if (this.MATRIX_OUTPUT != null) {
            IOUtil.assertFileIsWritable(this.MATRIX_OUTPUT);
        }
        if (this.INPUT_SAMPLE_MAP != null) {
            IOUtil.assertFileIsReadable(this.INPUT_SAMPLE_MAP);
        }
        if (this.INPUT_SAMPLE_FILE_MAP != null) {
            IOUtil.assertFileIsReadable(this.INPUT_SAMPLE_FILE_MAP);
        }
        if (this.SECOND_INPUT_SAMPLE_MAP != null) {
            IOUtil.assertFileIsReadable(this.SECOND_INPUT_SAMPLE_MAP);
        }
        HaplotypeMap map = new HaplotypeMap(this.HAPLOTYPE_MAP);
        FingerprintChecker checker = new FingerprintChecker(map);
        checker.setAllowDuplicateReads(this.ALLOW_DUPLICATE_READS);
        checker.setValidationStringency(this.VALIDATION_STRINGENCY);
        checker.setReferenceFasta(this.REFERENCE_SEQUENCE);
        ArrayList<String> extensions = new ArrayList<String>();
        extensions.add(SamReader.Type.BAM_TYPE.fileExtension());
        extensions.add(SamReader.Type.SAM_TYPE.fileExtension());
        extensions.add(SamReader.Type.CRAM_TYPE.fileExtension());
        extensions.addAll(Arrays.asList(IOUtil.VCF_EXTENSIONS));
        List<Path> inputPaths = IOUtil.getPaths(this.INPUT);
        List<Path> unrolledFiles = IOUtil.unrollPaths(inputPaths, extensions.toArray(new String[0]));
        if (this.TEST_INPUT_READABILITY) {
            IOUtil.assertPathsAreReadable(unrolledFiles);
        }
        List<Path> secondInputsPaths = IOUtil.getPaths(this.SECOND_INPUT);
        List<Path> unrolledFiles2 = IOUtil.unrollPaths(secondInputsPaths, extensions.toArray(new String[0]));
        if (this.TEST_INPUT_READABILITY) {
            IOUtil.assertPathsAreReadable(unrolledFiles2);
        }
        this.log.info("Fingerprinting " + unrolledFiles.size() + " INPUT files.");
        Map<FingerprintIdDetails, Fingerprint> uncappedFpMap = checker.fingerprintFiles(unrolledFiles, this.NUM_THREADS, 1, TimeUnit.DAYS);
        Map<FingerprintIdDetails, Fingerprint> fpMap = this.capFingerprints(uncappedFpMap);
        if (this.INPUT_SAMPLE_MAP != null) {
            this.remapFingerprints(fpMap, this.INPUT_SAMPLE_MAP, "INPUT_SAMPLE_MAP");
        }
        if (this.INPUT_SAMPLE_FILE_MAP != null) {
            this.remapFingerprintsFromFiles(fpMap, this.INPUT_SAMPLE_FILE_MAP);
        }
        ArrayList<CrosscheckMetric> metrics = new ArrayList<CrosscheckMetric>();
        if (this.SECOND_INPUT.isEmpty()) {
            this.log.info("Cross-checking all " + (Object)((Object)this.CROSSCHECK_BY) + " against each other");
            numUnexpected = this.crossCheckGrouped(fpMap, fpMap, metrics, Fingerprint.getFingerprintIdDetailsStringFunction(this.CROSSCHECK_BY), this.CROSSCHECK_BY);
        } else {
            this.log.info("Fingerprinting " + unrolledFiles2.size() + " SECOND_INPUT files.");
            Map<FingerprintIdDetails, Fingerprint> uncappedFpMap2 = checker.fingerprintFiles(unrolledFiles2, this.NUM_THREADS, 1, TimeUnit.DAYS);
            Map<FingerprintIdDetails, Fingerprint> fpMap2 = this.capFingerprints(uncappedFpMap2);
            if (this.SECOND_INPUT_SAMPLE_MAP != null) {
                this.remapFingerprints(fpMap2, this.SECOND_INPUT_SAMPLE_MAP, "SECOND_INPUT_SAMPLE_MAP");
            }
            switch (this.CROSSCHECK_MODE) {
                case CHECK_SAME_SAMPLE: {
                    this.log.info("Checking each sample in INPUT with the same sample in SECOND_INPUT.");
                    numUnexpected = this.checkFingerprintsBySample(fpMap, fpMap2, metrics);
                    break;
                }
                case CHECK_ALL_OTHERS: {
                    this.log.info("Checking each " + (Object)((Object)this.CROSSCHECK_BY) + " in INPUT with each " + (Object)((Object)this.CROSSCHECK_BY) + " in SECOND_INPUT.");
                    numUnexpected = this.crossCheckGrouped(fpMap, fpMap2, metrics, Fingerprint.getFingerprintIdDetailsStringFunction(this.CROSSCHECK_BY), this.CROSSCHECK_BY);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unpossible!");
                }
            }
        }
        if (metrics.stream().noneMatch(m -> m.LOD_SCORE != 0.0)) {
            this.log.error("No non-zero results found. This is likely an error. Probable cause: there are no reads or variants at fingerprinting sites ");
            return this.EXIT_CODE_WHEN_NO_VALID_CHECKS;
        }
        MetricsFile metricsFile = this.getMetricsFile();
        metricsFile.addAllMetrics(metrics);
        if (this.OUTPUT != null) {
            metricsFile.write(this.OUTPUT);
        } else {
            metricsFile.write(new OutputStreamWriter(System.out));
        }
        if (this.MATRIX_OUTPUT != null) {
            this.writeMatrix();
        }
        if (numUnexpected > 0) {
            this.log.warn(numUnexpected + " " + (Object)((Object)this.CROSSCHECK_BY) + "s did not relate as expected.");
            return this.EXIT_CODE_WHEN_MISMATCH;
        }
        this.log.info("All " + (Object)((Object)this.CROSSCHECK_BY) + "s are related as expected.");
        return 0;
    }

    private Map<FingerprintIdDetails, Fingerprint> capFingerprints(Map<FingerprintIdDetails, Fingerprint> fpMap) {
        return fpMap.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> {
            Fingerprint value = (Fingerprint)entry.getValue();
            Fingerprint cappedFp = new Fingerprint(value.getSample(), value.getSource(), value.getInfo());
            value.values().stream().map(probabilities -> new CappedHaplotypeProbabilities((HaplotypeProbabilities)probabilities, -this.MAX_EFFECT_OF_EACH_HAPLOTYPE_BLOCK)).forEach(cappedFp::add);
            return cappedFp;
        }));
    }

    private void remapFingerprints(Map<FingerprintIdDetails, Fingerprint> fpMap, File sampleMapFile, String inputFieldName) {
        Map<String, String> sampleMap = this.getStringStringMap(sampleMapFile, inputFieldName);
        Set samplesInFpMap = fpMap.keySet().stream().map(id -> id.sample).collect(Collectors.toSet());
        Set samplesNotInSampleMap = sampleMap.keySet().stream().filter(((Predicate<String>)samplesInFpMap::contains).negate()).collect(Collectors.toSet());
        if (!samplesNotInSampleMap.isEmpty()) {
            this.log.warn("Some samples in first column in the " + inputFieldName + " were not present as samples in fingerprinted file: [" + String.join((CharSequence)", ", samplesNotInSampleMap) + "].");
        }
        ArrayList<String> resultingSamples = new ArrayList<String>(samplesInFpMap);
        sampleMap.keySet().forEach(s -> {
            if (resultingSamples.remove(s)) {
                resultingSamples.add((String)sampleMap.get(s));
            }
        });
        if (CollectionUtil.makeSet(resultingSamples.toArray(new String[0])).size() != resultingSamples.size()) {
            HashSet duplicates = new HashSet();
            HashSet unique = new HashSet();
            resultingSamples.forEach(s -> {
                if (unique.add(s)) {
                    duplicates.add(s);
                }
            });
            throw new IllegalArgumentException("After applying the mapping found in the " + inputFieldName + " the resulting sample names must be unique when taken together with the remaining unmapped samples. Duplicates are: " + Arrays.toString(duplicates.toArray()));
        }
        HashSet<FingerprintIdDetails> ids = new HashSet<FingerprintIdDetails>(fpMap.keySet());
        ids.forEach(id -> {
            if (!sampleMap.containsKey(id.sample)) {
                return;
            }
            Fingerprint fingerprint = (Fingerprint)fpMap.remove(id);
            id.sample = (String)sampleMap.get(id.sample);
            fpMap.put((FingerprintIdDetails)id, fingerprint);
        });
    }

    private void remapFingerprintsFromFiles(Map<FingerprintIdDetails, Fingerprint> fpMap, File sampleMapFile) {
        Map<String, String> sampleMap = this.getStringStringMap(sampleMapFile, "INPUT_SAMPLE_FILE_MAP").entrySet().stream().collect(Collectors.toMap(e -> {
            try {
                return IOUtil.getPath((String)e.getValue()).toUri().toString();
            }
            catch (IOException e1) {
                throw new PicardException("Trouble reading file: " + (String)e.getValue(), e1);
            }
        }, Map.Entry::getKey));
        Set filesInFpMap = fpMap.keySet().stream().map(id -> id.file).collect(Collectors.toSet());
        Set sampleNotInFpMap = sampleMap.keySet().stream().filter(((Predicate<String>)filesInFpMap::contains).negate()).collect(Collectors.toSet());
        if (!sampleNotInFpMap.isEmpty()) {
            this.log.warn("Some samples from the first column in INPUT_SAMPLE_FILE_MAP were not found: " + Arrays.toString(sampleNotInFpMap.toArray()));
        }
        Map<String, List<FingerprintIdDetails>> fileFpDetailSetMap = fpMap.keySet().stream().collect(Collectors.groupingBy(s -> s.file));
        HashMap<String, String> fileSampleMap = new HashMap<String, String>();
        fileFpDetailSetMap.forEach((key, fingerprintIdDetails) -> {
            Set samples = fingerprintIdDetails.stream().map(id -> id.sample).collect(Collectors.toSet());
            if (samples.size() > 1) {
                throw new IllegalArgumentException("fingerprinting file (" + key + "in INPUT_SAMPLE_FILE_MAP contains multiple samples: " + String.join((CharSequence)"", samples));
            }
            fileSampleMap.put((String)key, ((FingerprintIdDetails)fingerprintIdDetails.get((int)0)).sample);
        });
        ArrayList<String> resultingSamples = new ArrayList<String>(filesInFpMap);
        fileSampleMap.forEach((f, id) -> {
            if (resultingSamples.remove(id)) {
                resultingSamples.add((String)sampleMap.get(f));
            }
        });
        if (CollectionUtil.makeSet(resultingSamples.toArray(new String[0])).size() != resultingSamples.size()) {
            HashSet duplicates = new HashSet();
            HashSet unique = new HashSet();
            resultingSamples.forEach(s -> {
                if (unique.add(s)) {
                    duplicates.add(s);
                }
            });
            throw new IllegalArgumentException("After applying the mapping found in the INPUT_SAMPLE_FILE_MAP the resulting sample names must be unique when taken together with the remaining unmapped samples. Duplicates are: " + Arrays.toString(duplicates.toArray()));
        }
        HashSet<FingerprintIdDetails> ids = new HashSet<FingerprintIdDetails>(fpMap.keySet());
        ids.forEach(id -> {
            if (!sampleMap.containsKey(id.file)) {
                return;
            }
            Fingerprint fingerprint = (Fingerprint)fpMap.remove(id);
            id.sample = (String)sampleMap.get(id.file);
            fpMap.put((FingerprintIdDetails)id, fingerprint);
        });
    }

    private Map<String, String> getStringStringMap(File sampleMapFile, String inputFieldName) {
        HashMap<String, String> sampleMap = new HashMap<String, String>();
        TabbedInputParser parser = new TabbedInputParser(false, sampleMapFile);
        for (Object[] strings : parser) {
            if (strings.length != 2) {
                throw new IllegalArgumentException("Each line of the " + inputFieldName + " must have exactly two strings separated by a tab. Found: [" + Arrays.toString(strings) + "] right before [" + parser.getCurrentLine() + "], in " + sampleMapFile.getAbsolutePath());
            }
            if (sampleMap.containsKey(strings[0])) {
                throw new IllegalArgumentException("Strings in first column of the " + inputFieldName + " must be unique. found [" + (String)strings[0] + "] twice. Right before [" + parser.getCurrentLine() + "] in " + sampleMapFile.getAbsolutePath());
            }
            sampleMap.put(strings[0], (String)strings[1]);
        }
        return sampleMap;
    }

    private void writeMatrix() {
        NumberFormat format = NumberFormat.getInstance();
        format.setMaximumFractionDigits(4);
        try (FileOutputStream stream = new FileOutputStream(this.MATRIX_OUTPUT);
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(stream));){
            writer.write(this.CROSSCHECK_BY.name());
            for (String rhsMatrixKey : this.rhsMatrixKeys) {
                writer.write('\t' + rhsMatrixKey);
            }
            writer.newLine();
            for (int row = 0; row < this.lhsMatrixKeys.size(); ++row) {
                writer.write(this.lhsMatrixKeys.get(row));
                for (double lod : this.crosscheckMatrix[row]) {
                    writer.write('\t' + format.format(lod));
                }
                writer.newLine();
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private int crossCheckGrouped(Map<FingerprintIdDetails, Fingerprint> lhsFingerprints, Map<FingerprintIdDetails, Fingerprint> rhsFingerprints, List<CrosscheckMetric> metrics, Function<FingerprintIdDetails, String> by, CrosscheckMetric.DataType type) {
        Map<FingerprintIdDetails, Fingerprint> lhsFingerprintsByGroup = Fingerprint.mergeFingerprintsBy(lhsFingerprints, by);
        Map<FingerprintIdDetails, Fingerprint> rhsFingerprintsByGroup = Fingerprint.mergeFingerprintsBy(rhsFingerprints, by);
        for (Map.Entry<FingerprintIdDetails, Fingerprint> pair : lhsFingerprintsByGroup.entrySet()) {
            if (pair.getValue().size() != 0) continue;
            this.log.warn(by.apply(pair.getKey()) + " was not fingerprinted in LEFT group.  It probably has no calls/reads overlapping fingerprinting sites.");
        }
        for (Map.Entry<FingerprintIdDetails, Fingerprint> pair : rhsFingerprintsByGroup.entrySet()) {
            if (pair.getValue().size() != 0) continue;
            this.log.warn(by.apply(pair.getKey()) + " was not fingerprinted in RIGHT group.  It probably has no calls/reads overlapping fingerprinting sites.");
        }
        if (this.MATRIX_OUTPUT != null) {
            this.crosscheckMatrix = new double[lhsFingerprintsByGroup.size()][];
            for (int row = 0; row < lhsFingerprintsByGroup.size(); ++row) {
                this.crosscheckMatrix[row] = new double[rhsFingerprintsByGroup.size()];
            }
            lhsFingerprintsByGroup.keySet().forEach(k -> this.lhsMatrixKeys.add(k.group));
            rhsFingerprintsByGroup.keySet().forEach(k -> this.rhsMatrixKeys.add(k.group));
        }
        return this.crossCheckFingerprints(lhsFingerprintsByGroup, rhsFingerprintsByGroup, type, metrics);
    }

    private int crossCheckFingerprints(Map<FingerprintIdDetails, Fingerprint> lhsFingerprints, Map<FingerprintIdDetails, Fingerprint> rhsFingerprints, CrosscheckMetric.DataType type, List<CrosscheckMetric> metrics) {
        int unexpectedResults = 0;
        long checksMade = 0L;
        int logEvery = 100000;
        ArrayList<FingerprintIdDetails> lhsFingerprintIdDetails = new ArrayList<FingerprintIdDetails>(lhsFingerprints.keySet());
        ArrayList<FingerprintIdDetails> rhsFingerprintIdDetails = new ArrayList<FingerprintIdDetails>(rhsFingerprints.keySet());
        long totalChecks = (long)lhsFingerprintIdDetails.size() * (long)rhsFingerprintIdDetails.size();
        for (int row = 0; row < lhsFingerprintIdDetails.size(); ++row) {
            FingerprintIdDetails lhsId = (FingerprintIdDetails)lhsFingerprintIdDetails.get(row);
            for (int col = 0; col < rhsFingerprintIdDetails.size(); ++col) {
                FingerprintIdDetails rhsId = (FingerprintIdDetails)rhsFingerprintIdDetails.get(col);
                boolean expectedToMatch = this.EXPECT_ALL_GROUPS_TO_MATCH || lhsId.sample.equals(rhsId.sample);
                MatchResults results = FingerprintChecker.calculateMatchResults(lhsFingerprints.get(lhsId), rhsFingerprints.get(rhsId), this.GENOTYPING_ERROR_RATE, this.LOSS_OF_HET_RATE, false, this.CALCULATE_TUMOR_AWARE_RESULTS);
                CrosscheckMetric.FingerprintResult result = this.getMatchResults(expectedToMatch, results);
                if (!this.OUTPUT_ERRORS_ONLY || result == CrosscheckMetric.FingerprintResult.INCONCLUSIVE || !result.isExpected().booleanValue()) {
                    metrics.add(this.getMatchDetails(result, results, lhsId, rhsId, type));
                }
                if (result != CrosscheckMetric.FingerprintResult.INCONCLUSIVE && !result.isExpected().booleanValue()) {
                    ++unexpectedResults;
                }
                if (this.crosscheckMatrix != null) {
                    this.crosscheckMatrix[row][col] = results.getLOD();
                }
                if (++checksMade % 100000L != 0L) continue;
                this.log.info("Compared " + checksMade + " of " + totalChecks);
            }
        }
        return unexpectedResults;
    }

    private int checkFingerprintsBySample(Map<FingerprintIdDetails, Fingerprint> fingerprints1, Map<FingerprintIdDetails, Fingerprint> fingerprints2, List<CrosscheckMetric> metrics) {
        int unexpectedResults = 0;
        Map<FingerprintIdDetails, Fingerprint> fingerprints1BySample = Fingerprint.mergeFingerprintsBy(fingerprints1, Fingerprint.getFingerprintIdDetailsStringFunction(CrosscheckMetric.DataType.SAMPLE));
        Map<FingerprintIdDetails, Fingerprint> fingerprints2BySample = Fingerprint.mergeFingerprintsBy(fingerprints2, Fingerprint.getFingerprintIdDetailsStringFunction(CrosscheckMetric.DataType.SAMPLE));
        Map<String, FingerprintIdDetails> sampleToDetail1 = fingerprints1BySample.keySet().stream().collect(Collectors.toMap(id -> id.group, id -> id));
        Map<String, FingerprintIdDetails> sampleToDetail2 = fingerprints2BySample.keySet().stream().collect(Collectors.toMap(id -> id.group, id -> id));
        HashSet<String> samples = new HashSet<String>();
        samples.addAll(sampleToDetail1.keySet());
        samples.addAll(sampleToDetail2.keySet());
        for (String sample : samples) {
            FingerprintIdDetails lhsID = sampleToDetail1.get(sample);
            FingerprintIdDetails rhsID = sampleToDetail2.get(sample);
            if (lhsID == null || rhsID == null) {
                this.log.error(String.format("sample %s is missing from %s group", sample, lhsID == null ? "LEFT" : "RIGHT"));
                ++unexpectedResults;
                continue;
            }
            Fingerprint lhsFP = fingerprints1BySample.get(lhsID);
            Fingerprint rhsFP = fingerprints2BySample.get(rhsID);
            if (lhsFP.size() == 0 || rhsFP.size() == 0) {
                this.log.error(String.format("sample %s from %s group was not fingerprinted.  Probably there are no reads/variants at fingerprinting sites.", sample, lhsFP.size() == 0 ? "LEFT" : "RIGHT"));
                ++unexpectedResults;
            }
            MatchResults results = FingerprintChecker.calculateMatchResults(lhsFP, rhsFP, this.GENOTYPING_ERROR_RATE, this.LOSS_OF_HET_RATE, false, this.CALCULATE_TUMOR_AWARE_RESULTS);
            CrosscheckMetric.FingerprintResult result = this.getMatchResults(true, results);
            if (!this.OUTPUT_ERRORS_ONLY || !result.isExpected().booleanValue()) {
                metrics.add(this.getMatchDetails(result, results, lhsID, rhsID, CrosscheckMetric.DataType.SAMPLE));
            }
            if (result != CrosscheckMetric.FingerprintResult.INCONCLUSIVE && !result.isExpected().booleanValue()) {
                ++unexpectedResults;
            }
            if (results.getLOD() != 0.0) continue;
            this.log.error("LOD score of zero found when checking sample fingerprints.  Probably there are no reads/variants at fingerprinting sites for one of the samples");
            ++unexpectedResults;
        }
        return unexpectedResults;
    }

    private CrosscheckMetric getMatchDetails(CrosscheckMetric.FingerprintResult matchResult, MatchResults results, FingerprintIdDetails leftPuDetails, FingerprintIdDetails rightPuDetails, CrosscheckMetric.DataType type) {
        CrosscheckMetric metric = new CrosscheckMetric();
        metric.LEFT_GROUP_VALUE = leftPuDetails.group;
        metric.RIGHT_GROUP_VALUE = rightPuDetails.group;
        metric.RESULT = matchResult;
        metric.LOD_SCORE = results.getLOD();
        metric.LOD_SCORE_TUMOR_NORMAL = results.getLodTN();
        metric.LOD_SCORE_NORMAL_TUMOR = results.getLodNT();
        metric.DATA_TYPE = type;
        metric.LEFT_RUN_BARCODE = leftPuDetails.runBarcode;
        metric.LEFT_LANE = leftPuDetails.runLane;
        metric.LEFT_MOLECULAR_BARCODE_SEQUENCE = leftPuDetails.molecularBarcode;
        metric.LEFT_LIBRARY = leftPuDetails.library;
        metric.LEFT_SAMPLE = leftPuDetails.sample;
        metric.LEFT_FILE = leftPuDetails.file;
        metric.RIGHT_RUN_BARCODE = rightPuDetails.runBarcode;
        metric.RIGHT_LANE = rightPuDetails.runLane;
        metric.RIGHT_MOLECULAR_BARCODE_SEQUENCE = rightPuDetails.molecularBarcode;
        metric.RIGHT_LIBRARY = rightPuDetails.library;
        metric.RIGHT_SAMPLE = rightPuDetails.sample;
        metric.RIGHT_FILE = rightPuDetails.file;
        return metric;
    }

    private CrosscheckMetric.FingerprintResult getMatchResults(boolean expectedToMatch, MatchResults results) {
        if (expectedToMatch) {
            if (results.getLOD() < this.LOD_THRESHOLD) {
                return CrosscheckMetric.FingerprintResult.UNEXPECTED_MISMATCH;
            }
            if (results.getLOD() > -this.LOD_THRESHOLD) {
                return CrosscheckMetric.FingerprintResult.EXPECTED_MATCH;
            }
            return CrosscheckMetric.FingerprintResult.INCONCLUSIVE;
        }
        if (results.getLOD() > -this.LOD_THRESHOLD) {
            return CrosscheckMetric.FingerprintResult.UNEXPECTED_MATCH;
        }
        if (results.getLOD() < this.LOD_THRESHOLD) {
            return CrosscheckMetric.FingerprintResult.EXPECTED_MISMATCH;
        }
        return CrosscheckMetric.FingerprintResult.INCONCLUSIVE;
    }
}

