/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;

@CommandLineProgramProperties(summary="Compare two metrics files.This tool compares the metrics and histograms generated from metric tools to determine if the generated results are identical.  Note that if there are differences in metric values, this tool describes those differences as the change of the second input metric relative to the first. <br /><br />  <h4>Usage example:</h4><pre>java -jar picard.jar CompareMetrics \\<br />      INPUT=metricfile1.txt \\<br />      INPUT=metricfile2.txt \\<br />      METRICS_TO_IGNORE=INSERT_LENGTH \\<br />      METRIC_ALLOWABLE_RELATIVE_CHANGE=HET_HOM_RATIO:0.0005 \\<br />      IGNORE_HISTOGRAM_DIFFERENCES=false</pre><hr />", oneLineSummary="Compare two metrics files.", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class CompareMetrics
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Compare two metrics files.";
    static final String USAGE_DETAIL = "This tool compares the metrics and histograms generated from metric tools to determine if the generated results are identical.  Note that if there are differences in metric values, this tool describes those differences as the change of the second input metric relative to the first. <br /><br />  <h4>Usage example:</h4><pre>java -jar picard.jar CompareMetrics \\<br />      INPUT=metricfile1.txt \\<br />      INPUT=metricfile2.txt \\<br />      METRICS_TO_IGNORE=INSERT_LENGTH \\<br />      METRIC_ALLOWABLE_RELATIVE_CHANGE=HET_HOM_RATIO:0.0005 \\<br />      IGNORE_HISTOGRAM_DIFFERENCES=false</pre><hr />";
    @Argument(shortName="I", doc="Metric files to compare.", minElements=2, maxElements=2)
    public List<File> INPUT;
    @Argument(shortName="O", doc="Output file to write comparison results to.", optional=true)
    public File OUTPUT;
    @Argument(shortName="MI", doc="Metrics to ignore. Any metrics specified here will be excluded from comparison by the tool.", optional=true)
    public List<String> METRICS_TO_IGNORE;
    @Argument(shortName="MARC", doc="Metric Allowable Relative Change. A colon separate pair of metric name and an absolute relative change.  For any metric specified here,  when the values are compared between the two files, the program will allow that much relative change between the  two values.", optional=true)
    public List<String> METRIC_ALLOWABLE_RELATIVE_CHANGE;
    @Argument(shortName="IHD", doc="Ignore any differences between the two metric file's histograms (useful if using the 'METRIC_ALLOWABLE_RELATIVE_CHANGE')", optional=true)
    public boolean IGNORE_HISTOGRAM_DIFFERENCES = false;
    private final List<String> differences = new ArrayList<String>();
    private String metricClassName = "Unknown";
    private static final Log log = Log.getInstance(CompareMetrics.class);
    protected final Map<String, Double> MetricToAllowableRelativeChange = new HashMap<String, Double>();

    @Override
    protected int doWork() {
        IOUtil.assertFilesAreReadable(this.INPUT);
        try {
            int retVal = this.compareMetricsFiles(this.INPUT.get(0), this.INPUT.get(1));
            String status = retVal == 0 ? "equal" : "NOT equal";
            log.info(this.metricClassName + " Metric files " + this.INPUT.get(0) + " and " + this.INPUT.get(1) + " are " + status);
            if (!this.differences.isEmpty()) {
                for (String difference : this.differences) {
                    log.error(difference);
                }
            }
            if (this.OUTPUT != null) {
                String header = "Comparison of " + this.metricClassName + " metrics between files " + this.INPUT.get(0).getAbsolutePath() + " and " + this.INPUT.get(1).getAbsolutePath() + "\n\nMetrics are " + status;
                CompareMetrics.writeTextToFile(this.OUTPUT, header, this.differences);
            }
            return retVal;
        }
        catch (Exception e) {
            throw new PicardException(e.getMessage());
        }
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> errs = new ArrayList<String>();
        if (this.OUTPUT != null) {
            IOUtil.assertFileIsWritable(this.OUTPUT);
        }
        if (this.METRIC_ALLOWABLE_RELATIVE_CHANGE != null) {
            for (String diffs : this.METRIC_ALLOWABLE_RELATIVE_CHANGE) {
                String[] pair = diffs.split(":");
                if (pair.length == 2) {
                    String name = pair[0];
                    try {
                        double value = Double.parseDouble(pair[1]);
                        if (value > 0.0) {
                            this.MetricToAllowableRelativeChange.put(name, value);
                            continue;
                        }
                        errs.add("Value for numeric component of Argument 'METRIC_ALLOWABLE_RELATIVE_CHANGE' must be > 0.0");
                    }
                    catch (NumberFormatException ne) {
                        errs.add("Invalid value for numeric component of Argument 'METRIC_ALLOWABLE_RELATIVE_CHANGE'");
                    }
                    continue;
                }
                errs.add("Invalid value for Argument 'METRIC_ALLOWABLE_RELATIVE_CHANGE'");
            }
        }
        if (errs.isEmpty()) {
            return null;
        }
        return errs.toArray(new String[0]);
    }

    private int compareMetricsFiles(File metricFile1, File metricFile2) throws IOException, IllegalAccessException {
        MetricsFile mf1 = new MetricsFile();
        MetricsFile mf2 = new MetricsFile();
        mf1.read(new FileReader(metricFile1));
        mf2.read(new FileReader(metricFile2));
        this.metricClassName = !mf1.getMetrics().isEmpty() ? ((MetricBase)mf1.getMetrics().get(0)).getClass().getName() : (!mf2.getMetrics().isEmpty() ? ((MetricBase)mf2.getMetrics().get(0)).getClass().getName() : "Unknown");
        boolean histogramsEqual = mf1.areHistogramsEqual(mf2);
        if (mf1.areMetricsEqual(mf2)) {
            if (histogramsEqual) {
                return 0;
            }
            if (this.IGNORE_HISTOGRAM_DIFFERENCES) {
                this.differences.add("Metrics Histograms differ, but the 'IGNORE_HISTOGRAM_DIFFERENCES' flag is set.");
                return 0;
            }
            this.differences.add("Metrics Histograms differ");
            return 1;
        }
        if (mf1.getMetrics().size() != mf2.getMetrics().size()) {
            this.differences.add("Number of metric rows differ between " + metricFile1.getAbsolutePath() + " and " + metricFile2.getAbsolutePath());
            return 1;
        }
        if (!((MetricBase)mf1.getMetrics().get(0)).getClass().equals(((MetricBase)mf2.getMetrics().get(0)).getClass())) {
            throw new PicardException("Metrics are of differing class between " + metricFile1.getAbsolutePath() + " and " + metricFile2.getAbsolutePath());
        }
        if (!mf1.getMetricsColumnLabels().equals(mf2.getMetricsColumnLabels())) {
            this.differences.add("Metric columns differ between " + metricFile1.getAbsolutePath() + " and " + metricFile2.getAbsolutePath());
            return 1;
        }
        CompareMetrics.validateMetricNames(mf1, metricFile1, this.METRICS_TO_IGNORE);
        CompareMetrics.validateMetricNames(mf1, metricFile1, this.MetricToAllowableRelativeChange.keySet());
        HashSet<String> metricsToIgnore = new HashSet<String>(this.METRICS_TO_IGNORE);
        Class<?> metricClass = ((MetricBase)mf1.getMetrics().get(0)).getClass();
        int retVal = 0;
        int rowNumber = -1;
        Field[] fields = metricClass.getFields();
        Iterator mf1Iterator = mf1.getMetrics().iterator();
        Iterator mf2Iterator = mf2.getMetrics().iterator();
        while (mf1Iterator.hasNext()) {
            ++rowNumber;
            MetricBase metric1 = (MetricBase)mf1Iterator.next();
            MetricBase metric2 = (MetricBase)mf2Iterator.next();
            for (Field field : fields) {
                if (metricsToIgnore.contains(field.getName())) continue;
                Object value1 = field.get(metric1);
                Object value2 = field.get(metric2);
                SimpleResult result = this.compareMetricValues(value1, value2, field.getName());
                if (result.equal) continue;
                retVal = 1;
                String diffString = "Row: " + rowNumber + " Metric: " + field.getName() + " values differ. Value1: " + value1 + " Value2: " + value2 + " " + result.description;
                this.differences.add(diffString);
            }
        }
        if (!this.IGNORE_HISTOGRAM_DIFFERENCES) {
            if (!histogramsEqual) {
                String diffString = "Metric Histograms differ";
                this.differences.add("Metric Histograms differ");
            }
            if (retVal == 0 && !histogramsEqual) {
                retVal = 1;
            }
        }
        return retVal;
    }

    protected SimpleResult compareMetricValues(Object value1, Object value2, String metricName) {
        boolean equal = true;
        String description = "";
        if (value1 == null || value2 == null) {
            if (value1 != null || value2 != null) {
                equal = false;
                description = "One of the values is null";
            }
        } else if (value1 instanceof Number) {
            double numValue1 = ((Number)value1).doubleValue();
            double numValue2 = ((Number)value2).doubleValue();
            double absoluteChange = 0.0;
            if (!Double.isNaN(numValue1) || !Double.isNaN(numValue2)) {
                absoluteChange = numValue2 - numValue1;
            }
            if (absoluteChange != 0.0) {
                double relativeChange;
                double d = relativeChange = numValue1 == 0.0 ? Double.MAX_VALUE : absoluteChange / numValue1;
                if (this.MetricToAllowableRelativeChange.containsKey(metricName)) {
                    double allowableRelativeChange = this.MetricToAllowableRelativeChange.get(metricName);
                    if (Math.abs(relativeChange) >= allowableRelativeChange) {
                        equal = false;
                        description = "Changed by " + absoluteChange + " (relative change of " + relativeChange + ") which is outside of the allowable relative change tolerance of " + allowableRelativeChange;
                    } else {
                        equal = true;
                        description = "Changed by " + absoluteChange + " (relative change of " + relativeChange + ") which is within the allowable relative change tolerance of " + allowableRelativeChange;
                    }
                } else {
                    equal = false;
                    description = "Changed by " + absoluteChange + " (relative change of " + relativeChange + ")";
                }
            }
        } else if (!value1.equals(value2)) {
            equal = false;
            description = "";
        }
        return new SimpleResult(equal, description);
    }

    private static void writeTextToFile(File output, String header, List<String> textLines) throws IOException {
        try (BufferedWriter writer = Files.newBufferedWriter(output.toPath(), new OpenOption[0]);){
            writer.write(header + "\n\n");
            writer.write(String.join((CharSequence)"\n", textLines));
        }
    }

    private static void validateMetricNames(MetricsFile<?, ?> metrics, File metricsFile, Collection<String> metricNames) {
        HashSet<String> metricsToIgnore = new HashSet<String>(metricNames);
        metricsToIgnore.removeAll(metrics.getMetricsColumnLabels());
        if (!metricsToIgnore.isEmpty()) {
            throw new PicardException("Metric(s) of the name: " + String.join((CharSequence)", ", metricsToIgnore) + " were not found in " + metricsFile.getAbsolutePath());
        }
    }

    static class SimpleResult {
        final boolean equal;
        final String description;

        public SimpleResult(boolean equal, String description) {
            this.equal = equal;
            this.description = description;
        }
    }
}

